<?php
// AsaasClient.php
class AsaasClient {
    private $apiKey;
    private $baseUrl;
    private $apiVersion = 'v3';

    public function __construct(string $apiKey, string $mode = 'sandbox') {
        $this->apiKey = $apiKey;
        $this->setEnvironment($mode);
    }

    /**
     * Cria um novo cliente na Asaas
     * @param array $data Dados do cliente
     * @return array Resposta da API
     */
    public function createCustomer(array $data): array {
        // Verificar se o campo 'name' foi informado
        if (empty($data['name'])) {
            throw new Exception("O campo 'name' é obrigatório para criar um cliente.");
        }

        // Depuração: Verificando os dados enviados
        error_log("Dados enviados para criação do cliente: " . json_encode($data));

        return $this->request('POST', 'customers', $data);
    }

    /**
     * Cria um pagamento via PIX
     * @param array $data Dados do pagamento
     * @return array Resposta da API
     */
    public function createPixPayment(array $data): array {
        $data['billingType'] = 'PIX';
        return $this->request('POST', 'payments', $data);
    }

    /**
     * Obtém os detalhes de um pagamento
     * @param string $paymentId ID do pagamento
     * @return array Resposta da API
     */
    public function getPayment(string $paymentId): array {
        return $this->request('GET', "payments/{$paymentId}");
    }

    /**
     * Gera QR Code para pagamento PIX
     * @param string $paymentId ID do pagamento
     * @return array Resposta da API
     */
    public function getPixQrCode(string $paymentId): array {
        return $this->request('GET', "payments/{$paymentId}/pixQrCode");
    }

    /**
     * Valida a assinatura do webhook
     * @param string $signature Assinatura recebida no header
     * @param string $webhookToken Token configurado
     * @param string $payload Corpo da requisição
     * @return bool
     */
    public function validateWebhookSignature(string $signature, string $webhookToken, string $payload): bool {
        $expectedSignature = hash_hmac('sha256', $payload, $webhookToken);
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Método genérico para requisições à API
     * @param string $method Método HTTP
     * @param string $endpoint Endpoint da API
     * @param array|null $data Dados para envio
     * @return array Resposta formatada
     * @throws Exception
     */
    private function request(string $method, string $endpoint, ?array $data = null): array {
        $url = "{$this->baseUrl}/{$this->apiVersion}/{$endpoint}";
        
        // Adicionando o cabeçalho User-Agent
        $headers = [
            'Content-Type: application/json',
            'access_token: ' . $this->apiKey,
            'User-Agent: Movcine/1.0' // Aqui você adiciona o User-Agent
        ];

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_CUSTOMREQUEST => strtoupper($method),
            CURLOPT_FAILONERROR => false,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 15
        ]);

        if ($data && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new Exception("Erro na comunicação com a API: {$error}");
        }

        $responseData = json_decode($response, true) ?? [];

        if ($httpCode >= 400) {
            $errorMessage = $responseData['errors'][0]['description'] ?? 'Erro desconhecido';
            throw new Exception("Erro {$httpCode} - {$errorMessage}");
        }

        return [
            'code' => $httpCode,
            'response' => $responseData
        ];
    }

    /**
     * Define o ambiente da API
     * @param string $mode Ambiente (sandbox/production)
     */
    private function setEnvironment(string $mode): void {
        $this->baseUrl = ($mode === 'production')
            ? 'https://api.asaas.com'
            : 'https://sandbox.asaas.com/api';
    }

    /**
     * Busca um cliente na Asaas pelo CPF ou e-mail
     * @param array $data Dados de busca (cpfCnpj ou email)
     * @return array Resposta da API
     */
    public function getCustomerByCpfOrEmail(array $data): array {
        // Verificar se forneceu CPF ou e-mail
        $endpoint = '';
        if (!empty($data['cpfCnpj'])) {
            $endpoint = "customers?cpfCnpj={$data['cpfCnpj']}";
        } elseif (!empty($data['email'])) {
            $endpoint = "customers?email={$data['email']}";
        } else {
            throw new Exception("É necessário fornecer um CPF ou e-mail.");
        }

        // Realiza a requisição para buscar o cliente
        $response = $this->request('GET', $endpoint);
        
        // Se não encontrar o cliente, cria um novo
        if ($response['code'] == 200 && empty($response['response']['data'])) {
            // Cria um cliente com os dados do formulário
            return $this->createCustomer($data); // Cria o cliente com os dados recebidos
        }

        return $response;
    }
}
