<?php
// admconfig.php
session_start();
require_once '../config.php';

// Verificar autenticação e permissões
if (!isset($_SESSION['admpag_user']) || $_SESSION['admpag_user']['access_level'] !== 'super') {
    header('Location: login.php');
    exit;
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();

        $id = $_POST['id'] ?? null;
        $data = [
            'username' => trim($_POST['username']),
            'email' => filter_var(trim($_POST['email']), FILTER_VALIDATE_EMAIL),
            'status' => isset($_POST['status']) ? 1 : 0,
            'access_level' => in_array($_POST['access_level'], ['basic', 'super']) ? $_POST['access_level'] : 'basic'
        ];

        // Validação
        if (empty($data['username']) || !$data['email']) {
            throw new Exception("Dados obrigatórios não preenchidos ou email inválido!");
        }

        // Se for novo usuário ou alteração de senha
        if (!empty($_POST['password'])) {
            if (strlen($_POST['password']) < 8) {
                throw new Exception("Senha deve ter no mínimo 8 caracteres!");
            }
            $data['password'] = password_hash($_POST['password'], PASSWORD_DEFAULT);
        }

        if ($id) {
            // Atualização
            $sql = "UPDATE admpag SET 
                    username = :username,
                    email = :email,
                    status = :status,
                    access_level = :access_level"
                    .(!empty($data['password']) ? ", password = :password" : "")."
                    WHERE id = :id";
        } else {
            // Inserção
            if (empty($data['password'])) {
                throw new Exception("Senha é obrigatória para novos usuários!");
            }
            $sql = "INSERT INTO admpag 
                    (username, email, password, status, access_level)
                    VALUES (:username, :email, :password, :status, :access_level)";
        }

        $stmt = $pdo->prepare($sql);
        if ($id) {
            $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        }
        $stmt->execute($data);

        $pdo->commit();
        $_SESSION['success'] = "Usuário salvo com sucesso!";
        header('Location: admconfig.php');
        exit;

    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Erro: " . $e->getMessage();
    }
}

// Processar exclusão
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    try {
        $stmt = $pdo->prepare("DELETE FROM admpag WHERE id = :id");
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $_SESSION['success'] = "Usuário excluído com sucesso!";
        header('Location: admconfig.php');
        exit;
    } catch (Exception $e) {
        $error = "Erro ao excluir: " . $e->getMessage();
    }
}

// Buscar usuários
$usuarios = $pdo->query("SELECT * FROM admpag ORDER BY username")->fetchAll();

// Modo edição
$editing = isset($_GET['edit']);
$usuarioEdicao = [];
if ($editing) {
    $stmt = $pdo->prepare("SELECT * FROM admpag WHERE id = ?");
    $stmt->execute([(int)$_GET['edit']]);
    $usuarioEdicao = $stmt->fetch();
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <title>Gerenciar Administradores</title>
    <!-- Links para os estilos do Bootstrap e Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <!-- Link para o favicon -->
    <link rel="icon" href="https://www.pokemongobrasil.com/wp-content/uploads/2015/09/cropped-favicon.png" type="image/x-icon">
    <style>
        .password-field { position: relative; }
        .password-toggle { position: absolute; right: 10px; top: 50%; transform: translateY(-50%); cursor: pointer; }
        .sidebar {
            height: 100vh;
            position: fixed;
            width: 280px;
            background-color: #f8f9fa;
            padding: 20px;
        }
        .main-content {
            margin-left: 280px;
            padding: 20px;
        }
    </style>
</head>
<body>
    <!-- Menu Lateral -->
    <?php include_once("menu-lateral.php"); ?>
    <!-- Fim do Menu Lateral -->
    
    <!-- Inicio do conteudo -->
    <div class="main-content">
        <div class="">
            <h2 class="mb-4">Gerenciar Usuários Administrativos</h2>

        <?php if(isset($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>
        
        <?php if(isset($_SESSION['success'])): ?>
            <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
            <?php unset($_SESSION['success']); ?>
        <?php endif; ?>

        <div class="card mb-4">
            <!-- Formulário de Edição/Criação -->
            <?php if($editing || isset($_GET['new'])): ?>
                <h4><?= $editing ? 'Editar Usuário' : 'Novo Usuário' ?></h4>
                <div class="card-body">
                    <form method="post">
                        <?php if($editing): ?>
                            <input type="hidden" name="id" value="<?= $usuarioEdicao['id'] ?>">
                        <?php endif; ?>
                        
                       <div class="row">
                            <div class="col-md-6 mb-3">
                            <label>Nome de Usuário*</label>
                            <input type="text" name="username" class="form-control" 
                                value="<?= htmlspecialchars($usuarioEdicao['username'] ?? '') ?>" required>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label>Email*</label>
                            <input type="email" name="email" class="form-control"
                                value="<?= htmlspecialchars($usuarioEdicao['email'] ?? '') ?>" required>
                        </div>
                        <div class="col-md-12 mb-3 password-field">
    <label for="password" class="form-label">Senha <?= $editing ? '(deixe em branco para não alterar)' : '*' ?></label>
    <input type="password" name="password" class="form-control" id="password" 
        <?= !$editing ? 'required' : '' ?>>
    <i class="password-toggle bi bi-eye-slash"></i>
</div>

                       </div>

                        

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label>Nível de Acesso</label>
                                <select name="access_level" class="form-select" required>
                                    <option value="basic" <?= ($usuarioEdicao['access_level'] ?? 'basic') === 'basic' ? 'selected' : '' ?>>Básico</option>
                                    <option value="super" <?= ($usuarioEdicao['access_level'] ?? '') === 'super' ? 'selected' : '' ?>>Super Admin</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label>Status</label>
                                <div class="form-check mt-2">
                                    <input type="checkbox" name="status" class="form-check-input" 
                                        <?= ($usuarioEdicao['status'] ?? 1) ? 'checked' : '' ?>>
                                    <label class="form-check-label">Ativo</label>
                                </div>
                            </div>
                        </div>

                        <button type="submit" class="btn btn-large btn-primary">
                            <?= $editing ? 'Atualizar' : 'Criar' ?>
                        </button>
                        <a href="admconfig.php" class="btn btn-secondary">Cancelar</a>
                    </form>
                </div>
            </div>
            <?php else: ?>
            <!-- Lista de Usuários -->
            <div class="card-header d-md-inline-flex align-items-center justify-content-between py-3 px-4">
                <h4>Usuários Cadastrados</h4>
                <a href="?new" class="btn btn-success d-inline-flex align-items-center">
                    <i class="bi bi-plus-lg"></i>
                    <span class="ps-1 text-nowrap d-none d-sm-block">Novo Usuário</span>
                </a>
            </div>

            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Usuário</th>
                        <th>Email</th>
                        <th>Nível</th>
                        <th>Status</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach($usuarios as $usuario): ?>
                        <tr>
                            <td><?= htmlspecialchars($usuario['username']) ?></td>
                            <td><?= htmlspecialchars($usuario['email']) ?></td>
                            <td>
                                <span class="badge text-uppercase bg-<?= $usuario['access_level'] === 'super' ? 'danger' : 'primary' ?>">
                                    <?php 
                                        $accessLevels = [
                                            'basic' => 'Básico',
                                            'super' => 'Super Admin'
                                        ];
                                        echo isset($accessLevels[strtolower($usuario['access_level'])]) ? $accessLevels[strtolower($usuario['access_level'])] : 'Nível Desconhecido';
                                    ?>
                                </span>
                            </td>
                            <td>
                                <?php if($usuario['status']): ?>
                                    <span class="badge bg-success">Ativo</span>
                                <?php else: ?>
                                    <span class="badge bg-secondary">Inativo</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="?edit=<?= $usuario['id'] ?>" class="btn btn-sm btn-warning">Editar</a>
                                <a href="?delete=<?= $usuario['id'] ?>" 
                                    class="btn btn-sm btn-danger"
                                    onclick="return confirm('Tem certeza que deseja excluir este usuário?')">
                                    Excluir
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>
    </div>
    <!-- Fim do conteudo -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle de visibilidade da senha
        document.querySelectorAll('.password-toggle').forEach(icon => {
            icon.addEventListener('click', function() {
                const input = this.previousElementSibling;
                const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
                input.setAttribute('type', type);
                this.classList.toggle('bi-eye');
                this.classList.toggle('bi-eye-slash');
            });
        });
    </script>
</body>
</html>
