<?php
// plans.php
session_start();
require_once '../config.php';

// Verificar autenticação
if (!isset($_SESSION['admpag_user'])) {
    header('Location: login.php');
    exit;
}

// Processar formulário CRUD
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();

        $id = $_POST['id'] ?? null;
        $data = [
            'name' => trim($_POST['name']),
            'price' => (float) str_replace(',', '.', $_POST['price']),
            'duration_days' => (int) $_POST['duration_days']
        ];

        // Validação básica
        if (empty($data['name']) || $data['price'] <= 0 || $data['duration_days'] < 1) {
            throw new Exception("Dados inválidos!");
        }

        if ($id) {
            // Atualizar plano
            $stmt = $pdo->prepare("UPDATE planospix SET 
                name = :name,
                price = :price,
                duration_days = :duration_days,
                updated_at = NOW()
                WHERE id = :id");
                
            $data['id'] = $id;
        } else {
            // Criar novo plano
            $stmt = $pdo->prepare("INSERT INTO planospix 
                (name, price, duration_days, created_at, updated_at)
                VALUES (:name, :price, :duration_days, NOW(), NOW())");
        }

        $stmt->execute($data);
        $pdo->commit();
        
        header('Location: plans.php');
        exit;

    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Erro: " . $e->getMessage();
    }
}

// Processar exclusão
if (isset($_GET['delete'])) {
    $id = (int) $_GET['delete'];
    
    try {
        $stmt = $pdo->prepare("DELETE FROM planospix WHERE id = ?");
        $stmt->execute([$id]);
        header('Location: plans.php');
        exit;
    } catch (Exception $e) {
        $error = "Erro ao excluir: " . $e->getMessage();
    }
}

// Buscar planos
$plans = $pdo->query("SELECT * FROM planospix ORDER BY price ASC")->fetchAll();

// Verificar modo edição
$editing = isset($_GET['edit']);
$planToEdit = [];
if ($editing) {
    $stmt = $pdo->prepare("SELECT * FROM planospix WHERE id = ?");
    $stmt->execute([(int) $_GET['edit']]);
    $planToEdit = $stmt->fetch();
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <title>Gerenciar Planos</title>
    <!-- Links para os estilos do Bootstrap e Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <!-- Link para o favicon -->
    <link rel="icon" href="https://www.pokemongobrasil.com/wp-content/uploads/2015/09/cropped-favicon.png" type="image/x-icon">
    <style>
        .sidebar {
            height: 100vh;
            position: fixed;
            width: 280px;
            background-color: #f8f9fa;
            padding: 20px;
        }
        .main-content {
            margin-left: 280px;
            padding: 20px;
        }
        .table-actions {
            width: 150px;
        }
    </style>
</head>
<body>
    <!-- Menu Lateral -->
    <?php
        include_once("menu-lateral.php");
    ?>
    <!-- Fim do Menu Lateral -->

    <div class="main-content">
        <!-- Mensagens de Erro/Sucesso -->
        <?php if(isset($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>

        <!-- Formulário de Edição/Criação -->
        <?php if($editing || isset($_GET['new'])): ?>
            <div class="form-container">
                <h2><?= $editing ? 'Editar Plano' : 'Novo Plano' ?></h2>
                <form method="post">
                    <?php if($editing): ?>
                        <input type="hidden" name="id" value="<?= $planToEdit['id'] ?>">
                    <?php endif; ?>
                    
                    <div class="mb-3">
                        <label class="form-label">Nome do Plano</label>
                        <input type="text" name="name" class="form-control" 
                            value="<?= htmlspecialchars($planToEdit['name'] ?? '') ?>" required>
                    </div>

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Preço (R$)</label>
                            <input type="number" step="0.01" name="price" class="form-control"
                                value="<?= number_format($planToEdit['price'] ?? 0, 2, '.', '') ?>" required>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Duração (dias)</label>
                            <input type="number" name="duration_days" class="form-control"
                                value="<?= $planToEdit['duration_days'] ?? '' ?>" min="1" required>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <?= $editing ? 'Atualizar' : 'Criar' ?>
                        </button>
                        <a href="plans.php" class="btn btn-secondary">Cancelar</a>
                    </div>
                </form>
            </div>

        <!-- Lista de Planos -->
        <?php else: ?>
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1>Planos Disponíveis</h1>
                <a href="?new" class="btn btn-success">
                    <i class="bi bi-plus-lg"></i> Novo Plano
                </a>
            </div>

            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Preço</th>
                        <th>Duração</th>
                        <th class="table-actions">Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach($plans as $plan): ?>
                        <tr>
                            <td>R$ <?= number_format($plan['price'], 2, ',', '.') ?></td>
                            <td><?= $plan['duration_days'] ?> dias</td>
                            <td>
                                <a href="?edit=<?= $plan['id'] ?>" class="btn btn-sm btn-warning">Editar</a>
                                <a href="?delete=<?= $plan['id'] ?>" 
                                    class="btn btn-sm btn-danger"
                                    onclick="return confirm('Tem certeza que deseja excluir?')">
                                    Excluir
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>