<?php
session_start();
require_once 'config.php';
require_once 'AsaasClient.php';

// Função para validar CPF
function isValidCPF($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    if (strlen($cpf) != 11) {
        return false;
    }
    if (preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$t] != $d) {
            return false;
        }
    }
    return true;
}

// Função para validar CNPJ
function isValidCNPJ($cnpj) {
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
    if (strlen($cnpj) != 14) {
        return false;
    }
    if (preg_match('/(\d)\1{13}/', $cnpj)) {
        return false;
    }
    for ($t = 12; $t < 14; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cnpj[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cnpj[$t] != $d) {
            return false;
        }
    }
    return true;
}

// Função para validar CPF ou CNPJ
function isValidCPFOrCNPJ($cpfCnpj) {
    if (strlen($cpfCnpj) == 11) {
        return isValidCPF($cpfCnpj);
    } elseif (strlen($cpfCnpj) == 14) {
        return isValidCNPJ($cpfCnpj);
    } else {
        return false;
    }
}

// Função para validar telefone (somente dígitos, 10 ou 11 caracteres)
function isValidPhone($phone) {
    return preg_match('/^\d{10,11}$/', $phone);
}

// Verificar se o botão 'Assinar' foi pressionado
if (!isset($_POST['Assinar'])) {
    echo json_encode([
        'success' => false,
        'error_message' => "Botão 'Assinar' não foi pressionado ou os dados não são válidos."
    ]);
    exit;
}
unset($_POST['Assinar']);

// Recebendo e validando os dados do formulário
$email       = filter_input(INPUT_POST, 'email', FILTER_VALIDATE_EMAIL);
$plan_id     = filter_input(INPUT_POST, 'plan_id', FILTER_VALIDATE_INT);
$mobilePhone = filter_input(INPUT_POST, 'mobilePhone', FILTER_SANITIZE_NUMBER_INT); 
$cpfCnpj     = filter_input(INPUT_POST, 'cpfCnpj', FILTER_SANITIZE_NUMBER_INT);

if (!$email || !$plan_id || empty($mobilePhone) || empty($cpfCnpj)) {
    echo json_encode([
        'success' => false,
        'error_message' => "Dados inválidos ou campos obrigatórios faltando.",
        'email'       => $email,
        'plan_id'     => $plan_id,
        'mobilePhone' => $mobilePhone,
        'cpfCnpj'     => $cpfCnpj,
    ]);
    exit;
}

if (!isValidPhone($mobilePhone)) {  
    echo json_encode([
        'success' => false,
        'error_message' => "Número de telefone inválido."
    ]);
    exit;
}

if (!isValidCPFOrCNPJ($cpfCnpj)) {  
    echo json_encode([
        'success' => false,
        'error_message' => "CPF ou CNPJ inválido."
    ]);
    exit;
}

try {
    // Buscar usuário na tabela "users" pelo e-mail
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    if (!$user) {
        // Se o usuário não existir, insere um novo registro
        $stmt = $pdo->prepare("INSERT INTO users (email) VALUES (?)");
        $stmt->execute([$email]);
        $user_id = $pdo->lastInsertId();
    } else {
        $user_id = $user['id'];
    }

    // Verificar se já existe um registro na tabela "pix" para este usuário
    $stmt = $pdo->prepare("SELECT * FROM pix WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $pixData = $stmt->fetch();

    if ($pixData) {
        $mobilePhone = $pixData['mobilePhone'] ?? $mobilePhone;
        $cpfCnpj     = $pixData['cpfCnpj'] ?? $cpfCnpj;
    } else {
        $stmt = $pdo->prepare("INSERT INTO pix (user_id, mobilePhone, cpfCnpj) VALUES (?, ?, ?)");
        $stmt->execute([$user_id, $mobilePhone, $cpfCnpj]);
    }

    // Buscar informações do plano selecionado na tabela "planospix"
    $stmt = $pdo->prepare("SELECT * FROM planospix WHERE id = ?");
    $stmt->execute([$plan_id]);
    $plan = $stmt->fetch();

    if (!$plan) {
        echo json_encode([
            'success' => false,
            'error_message' => "Plano selecionado é inválido."
        ]);
    }

    // Buscar configurações de pagamento na tabela "asaas_settings"
    $asaasConfig = $pdo->query("SELECT * FROM asaas_settings WHERE id = 1")->fetch();
    if (!$asaasConfig) {
        echo json_encode([
            'success' => false,
            'error_message' => "Configurações de pagamento não encontradas."
        ]);
    }

    // Criar cliente na Asaas
    $asaas = new AsaasClient($asaasConfig['api_key'], $asaasConfig['mode']);
    $customerData = [
        'name'        => (!empty($user['name']) ? $user['name'] : $email),
        'email'       => $email,
        'mobilePhone' => $mobilePhone,
        'cpfCnpj'     => $cpfCnpj
    ];

    $customerResponse = $asaas->createCustomer($customerData);
    if ($customerResponse['code'] !== 200 || !isset($customerResponse['response']['id'])) {
        echo json_encode([
            'success' => false,
            'error_message' => "Erro ao criar cliente na plataforma de pagamentos: " . json_encode($customerResponse)
        ]);
    }

    // Criar cobrança PIX
    $paymentData = [
        'customer'    => $customerResponse['response']['id'],
        'billingType' => 'PIX',
        'value'       => (float) $plan['price'],
        'dueDate'     => date('Y-m-d', strtotime('+3 days'))
    ];

    $paymentResponse = $asaas->createPixPayment($paymentData);
    
    // Log para depuração: registra a resposta completa da API
    error_log("Resposta da API (createPixPayment): " . json_encode($paymentResponse));

    if ($paymentResponse['code'] !== 200 || !isset($paymentResponse['response']['id'])) {
        echo json_encode([
            'success' => false,
            'error_message' => "Erro ao gerar cobrança PIX: " . json_encode($paymentResponse)
        ]);
    }

    /* Verificar se a resposta contém os campos do QR Code e payload.
       Se não estiverem presentes, usa o campo invoiceUrl para gerar um QR Code.
    */
    if (isset($paymentResponse['response']['qr_code']) && !empty($paymentResponse['response']['qr_code'])) {
        $qr_code = $paymentResponse['response']['qr_code'];
        $payload = $paymentResponse['response']['payload'];
    } elseif (isset($paymentResponse['response']['qrCode']) && !empty($paymentResponse['response']['qrCode'])) {
        $qr_code = $paymentResponse['response']['qrCode'];
        $payload = $paymentResponse['response']['payload'];
    } elseif (isset($paymentResponse['response']['qrcodeBase64']) && !empty($paymentResponse['response']['qrcodeBase64'])) {
        $qr_code = $paymentResponse['response']['qrcodeBase64'];
        $payload = $paymentResponse['response']['payload'];
    } elseif (isset($paymentResponse['response']['invoiceUrl']) && !empty($paymentResponse['response']['invoiceUrl'])) {
        // Se não houver QR Code, gera um a partir da invoiceUrl usando um serviço de QR Code gratuito
        $invoiceUrl = $paymentResponse['response']['invoiceUrl'];
        $qr_code = "https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=" . urlencode($invoiceUrl);
        $payload = $invoiceUrl; // Fallback: utiliza o invoiceUrl como payload
    } else {
        echo json_encode([
            'success' => false,
            'error_message' => "QR Code não encontrado na resposta do pagamento. Resposta completa: " . json_encode($paymentResponse)
        ]);
    }

    // Inserir os dados do pagamento na tabela "pix" (sem o campo plan_id)
    $stmt = $pdo->prepare("INSERT INTO pix 
        (user_id, customer_id, cpfCnpj, mobilePhone, payment_id, status, qr_code, payload, due_date)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $user_id,
        $customerResponse['response']['id'],
        $cpfCnpj,
        $mobilePhone,
        $paymentResponse['response']['id'],
        $paymentResponse['response']['status'],
        $qr_code,
        $payload,
        date('Y-m-d H:i:s', strtotime($paymentResponse['response']['dueDate']))
    ]);

    // Redireciona para a página de QR Code, passando também o plan_id via GET
    header('Location: pagamento_qrcode.php?payment_id=' . $paymentResponse['response']['id'] . '&plan_id=' . $plan_id);
exit();
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error_message' => "Erro desconhecido: " . $e->getMessage()
    ]);
}
//exit;
