<?php
session_start();
require_once __DIR__ . '/../config.php'; // Caminho corrigido para o config.php
require_once __DIR__ . '/../lib/MercadoPagoClient.php'; // Caminho corrigido para a classe do Mercado Pago


// Funções de validação
function validarCPF($cpf) {
    $cpf = preg_replace('/\D/', '', $cpf);
    if (strlen($cpf) != 11) return false;
    // Lógica de validação de CPF (verificar dígitos verificadores)
    return true;
}

function validarCNPJ($cnpj) {
    $cnpj = preg_replace('/\D/', '', $cnpj);
    if (strlen($cnpj) != 14) return false;
    // Lógica de validação de CNPJ (verificar dígitos verificadores)
    return true;
}

// Gerar token CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); // Gerar um novo token CSRF
}

// Verificar se há mensagens de erro na sessão
if (isset($_SESSION['error'])) {
    $error = $_SESSION['error'];
    unset($_SESSION['error']);
}

// Buscar planos disponíveis na tabela "planospix"
try {
    $stmt = $pdo->query("SELECT * FROM planospix ORDER BY price ASC");
    $plans = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $plans = [];
    $error = "Erro ao buscar planos: " . $e->getMessage();
}

// Variável para verificar se o email já existe
$emailExistente = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verificação CSRF
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error = "Token CSRF inválido.";
    }

    // Recebe e sanitiza os dados
    $email = isset($_POST['email']) ? filter_var($_POST['email'], FILTER_SANITIZE_EMAIL) : null;
    $cpfCnpj = isset($_POST['cpfCnpj']) ? filter_var($_POST['cpfCnpj'], FILTER_SANITIZE_SPECIAL_CHARS) : null;
    $telefone = isset($_POST['tel']) ? filter_var($_POST['tel'], FILTER_SANITIZE_SPECIAL_CHARS) : null;

    // Verificação simples dos campos
    if (empty($cpfCnpj) || empty($telefone)) {
        $error = "CPF/CNPJ ou telefone não podem ser vazios.";
    } elseif (!preg_match('/^\d{11}$/', $telefone)) {
        $error = "Número de telefone inválido.";
    } elseif (!preg_match('/^\d{11,14}$/', $cpfCnpj)) {
        $error = "CPF/CNPJ inválido.";
    } elseif ($email) {
        // Verifica se o e-mail existe na tabela users
        $stmt = $pdo->prepare("SELECT * FROM users WHERE email = :email");
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        $emailExistente = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($emailExistente['email'] === $email) {
            $_SESSION['user_id'] = $emailExistente['id'];
        } else {
            $error = "E-mail não encontrado no sistema.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <title>Assinatura</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .plan-card {
            transition: transform 0.3s;
            border: 1px solid #2b2d42;
        }
        .plan-card:hover {
            transform: translateY(-10px);
            border-color: #ef233c;
        }
        .pix-instructions {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
        }
        .plan-card .btn:hover {
            background-color: #ef233c;
            color: white;
            transition: background-color 0.3s;
        }
    </style>
</head>
<body class="bg-dark text-light">
    <div class="container py-5">
        <h1 class="text-center mb-5">Escolha Seu Plano</h1>

        <!-- Exibe mensagem de erro, se houver -->
        <?php if (isset($error)): ?>
            <div class="alert alert-danger text-center"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>

        <!-- Formulário para verificação do e-mail -->
        <div class="row justify-content-center mb-5">
            <div class="col-md-12 col-sm-12 col-lg-7">
                <form method="post" action="" class="card bg-secondary text-white p-4">
                    <div class="row">
                        <div class="mb-3 col-md-6">
                            <label for="cpfCnpj" class="form-label">Digite CPF ou CNPJ cadastrado</label>
                            <input type="text" placeholder="Digite CPF/CNPJ" id="cpfCnpj" name="cpfCnpj" class="form-control" required>
                        </div>
                        <div class="mb-3 col-md-6">
                            <label for="telefone" class="form-label">Digite seu WhatsApp cadastrado</label>
                            <input type="tel" placeholder="WhatsApp" id="telefone" name="tel" class="form-control" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="email" class="form-label">Digite seu e-mail cadastrado</label>
                        <input type="email" placeholder="Digite seu email" id="email" name="email" class="form-control" required>
                        <div id="emailHelp" class="form-text text-light">Verificaremos se você já possui cadastro</div>
                    </div>
                    <input type="hidden" name="csrf_token" value="<?= $_SESSION['csrf_token'] ?>">
                    <button type="submit" class="btn btn-primary w-100">Verificar</button>
                </form>
            </div>
        </div>

        <!-- Se o e-mail foi verificado, exibe os planos -->
        <div id="emailResult" class="mt-3">
            <?php if ($emailExistente): ?>
                <div class="alert alert-success">E-mail verificado com sucesso!</div>
                <div id="planosContainer" class="mt-3">
                    <h2 class="text-center mb-4">Escolha Seu Plano</h2>
                    <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 row-cols-xl-4 g-4">
                        <?php foreach ($plans as $plan): ?>
                            <div class="col">
                                <form method="post" action="processamento/processa_pagamento_mercado_pago.php" class="plan-form">
                                    <input type="hidden" name="mobilePhone" value="<?= htmlspecialchars($telefone ?? '') ?>"> 
                                    <input type="hidden" name="cpfCnpj" value="<?= htmlspecialchars($cpfCnpj ?? '') ?>"> 
                                    <input type="hidden" name="email" value="<?= htmlspecialchars($email) ?>"> 
                                    <input type="hidden" name="plan_id" value="<?= $plan['id'] ?>"> 
                                    <div class="card h-100 bg-secondary text-white plan-card">
                                        <div class="card-body text-center">
                                            <h3><?= htmlspecialchars($plan['name']) ?></h3>
                                            <div class="display-4 mb-3">R$ <?= number_format($plan['price'], 2, ',', '.') ?></div>
                                            <button type="submit" name="Assinar" class="btn btn-danger btn-lg w-100">Assinar</button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && !$emailExistente): ?>
                <div class="alert alert-danger">E-mail não cadastrado. <a href="/cadastro" class="alert-link">Criar conta</a></div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
