<?php
session_start();
require_once __DIR__ . '/../../config.php'; // Caminho do config.php
require_once __DIR__ . '/../../lib/MercadoPagoClient.php'; // Classe Mercado Pago


// Função para validar CPF
function validarCPF($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    if (strlen($cpf) != 11) return false;

    // Calcula os dígitos verificadores
    for ($t = 9; $t < 11; $t++) {
        $d = 0;
        for ($c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) return false;
    }
    return true;
}

// Função para validar CNPJ
function validarCNPJ($cnpj) {
    $cnpj = preg_replace('/\D/', '', $cnpj);
    if (strlen($cnpj) != 14) return false;

    // Implementação simples de validação de CNPJ com dígitos verificadores
    $somatorio1 = 0;
    $somatorio2 = 0;
    $peso1 = [5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2];
    $peso2 = [6, 5, 4, 3, 2, 9, 8, 7, 6, 5, 4, 3, 2];

    // Calculando o primeiro dígito verificador
    for ($i = 0; $i < 12; $i++) {
        $somatorio1 += $cnpj[$i] * $peso1[$i];
    }
    $resto1 = $somatorio1 % 11;
    $digito1 = $resto1 < 2 ? 0 : 11 - $resto1;

    // Calculando o segundo dígito verificador
    for ($i = 0; $i < 13; $i++) {
        $somatorio2 += $cnpj[$i] * $peso2[$i];
    }
    $resto2 = $somatorio2 % 11;
    $digito2 = $resto2 < 2 ? 0 : 11 - $resto2;

    // Verifica os dígitos
    if ($cnpj[12] != $digito1 || $cnpj[13] != $digito2) {
        return false;
    }
    return true;
}

// Verifica se a requisição é POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Verifica a conexão com o banco de dados
        if (!isset($pdo) || !$pdo) {
            throw new Exception("Erro de conexão com o banco de dados.");
        }

        // Recebe e sanitiza os dados
        $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
        $cpfCnpj = preg_replace('/\D/', '', $_POST['cpfCnpj']); // Remove não numéricos
        $telefone = preg_replace('/\D/', '', $_POST['mobilePhone']); // Remove não numéricos
        $plan_id = filter_input(INPUT_POST, 'plan_id', FILTER_SANITIZE_NUMBER_INT);

        unset($_POST["Assinar"]);

        // Validação dos dados
        if (empty($email) || empty($cpfCnpj) || empty($telefone) || empty($plan_id)) {
            throw new Exception("Dados incompletos. Preencha todos os campos.");
        }

        // Validação do CPF/CNPJ
        if (strlen($cpfCnpj) == 11 && !validarCPF($cpfCnpj)) {
            throw new Exception("CPF inválido.");
        } elseif (strlen($cpfCnpj) == 14 && !validarCNPJ($cpfCnpj)) {
            throw new Exception("CNPJ inválido.");
        }

        // Buscar plano no banco de dados
        $stmt = $pdo->prepare("SELECT * FROM planospix WHERE id = :id");
        $stmt->bindParam(':id', $plan_id, PDO::PARAM_INT);
        $stmt->execute();
        $plan = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$plan) {
            throw new Exception("Plano não encontrado no banco.");
        }

        // Criar pagamento via PIX Mercado Pago
        $mercadoPagoClient = new MercadoPagoClient(MERCADO_PAGO_ACCESS_TOKEN);
        $payment = $mercadoPagoClient->criarPagamentoPix(
            $plan['price'], // Valor do plano
            $email,         // E-mail do pagador
            $cpfCnpj,       // CPF/CNPJ do pagador
            $plan['name']   // Descrição do pagamento
        );
        
        
        

        // Verifica se a API retornou um pagamento válido
        if (!isset($payment->id) || empty($payment->id)) {
            $errorDetails = isset($payment->message) ? $payment->message : 'Erro desconhecido na resposta da API.';
            error_log("Falha ao criar pagamento PIX. Resposta da API: " . json_encode($payment));
            throw new Exception("Falha ao criar pagamento PIX. Detalhes: " . $errorDetails);
        }

        // Salvar pagamento no banco de dados
        $stmt = $pdo->prepare("INSERT INTO pagamentos (email, cpfcnpj, telefone, plano_id, payment_id, status) 
                               VALUES (:email, :cpfcnpj, :telefone, :plano_id, :payment_id, :status)");
        
        $stmt->execute([
            ':email' => $email,
            ':cpfcnpj' => $cpfCnpj,
            ':telefone' => $telefone,
            ':plano_id' => $plan_id,
            ':payment_id' => $payment->id,
            ':status' => $payment->status
        ]);
        
        // Verifica se o pagamento foi salvo corretamente no banco
        if ($stmt->rowCount() == 0) {
            throw new Exception("Erro ao salvar pagamento no banco de dados.");
        }

        // Salva QR Code na sessão para exibir na página de sucesso
        $_SESSION['qr_code_url'] = $payment->qr_code ?? null;

        // Redireciona para a página de sucesso
        header("Location: /public/sucesso.php?payment_id=" . urlencode($payment->id));
        exit();

    } catch (Exception $e) {
        // Registra erro no log
        error_log("Erro no pagamento PIX: " . $e->getMessage());

        // Armazena mensagem de erro na sessão
        $_SESSION['error'] = $e->getMessage();

        // Redireciona para a página de erro
        /*header("Location: /public/falha.php");
        exit();*/
    }
} else {
    // Se não for POST, redireciona para a página inicial
    header("Location: /public/index.php");
    exit();
}
